---
name: Dependency Auditor
description: Check dependencies for major version updates, security vulnerabilities, and breaking changes. Use when checking dependencies, auditing packages, reviewing outdated dependencies, or when user mentions "dependencies", "outdated packages", "npm outdated", "security audit", "check updates", "package vulnerabilities".
allowed-tools: Bash, Read, Write
---

# Dependency Auditor

Intelligent dependency analysis that identifies outdated packages, security vulnerabilities, breaking changes, and provides actionable update recommendations with risk assessment.

## Core Capabilities

1. **Version Analysis**: Detect outdated packages and categorize by update type (major/minor/patch)
2. **Security Scanning**: Identify known vulnerabilities
3. **Breaking Change Detection**: Warn about major version updates
4. **Update Recommendations**: Prioritized suggestions with risk levels
5. **Compatibility Check**: Ensure peer dependencies are satisfied

## Instructions

### Step 1: Check Outdated Packages

```bash
# Get list of outdated packages
npm outdated --json > /tmp/outdated.json || echo "{}"

# Also check with npm-check-updates for more details
npx npm-check-updates --json > /tmp/ncu.json || echo "{}"
```

Parse the output to categorize updates:
- **Patch**: Bug fixes (1.0.0 → 1.0.1) - Low risk
- **Minor**: New features, backward compatible (1.0.0 → 1.1.0) - Low-Medium risk
- **Major**: Breaking changes (1.0.0 → 2.0.0) - High risk

### Step 2: Security Audit

```bash
# Run npm audit
npm audit --json > /tmp/audit.json

# Get severity counts
npm audit --json | jq '.metadata'
```

Categorize vulnerabilities:
- **Critical**: Immediate action required
- **High**: Address soon
- **Moderate**: Plan to fix
- **Low**: Monitor

### Step 3: Analyze Each Outdated Package

For packages with major version updates:

```bash
# Check changelog (common locations)
npm view <package-name> homepage
npm view <package-name> repository.url

# Check for BREAKING CHANGES in recent commits
# Check migration guides
```

### Step 4: Generate Risk Assessment

```typescript
interface PackageUpdate {
  name: string;
  current: string;
  wanted: string;  // Satisfies package.json range
  latest: string;
  updateType: 'major' | 'minor' | 'patch';
  hasVulnerabilities: boolean;
  vulnerabilities: {
    severity: 'critical' | 'high' | 'moderate' | 'low';
    count: number;
  }[];
  riskLevel: 'low' | 'medium' | 'high' | 'critical';
}

function calculateRisk(pkg: PackageUpdate): string {
  // Critical: Security vulnerabilities
  if (pkg.vulnerabilities.some(v => v.severity === 'critical')) {
    return 'critical';
  }

  // High: Major version update OR high severity vulnerabilities
  if (pkg.updateType === 'major' ||
      pkg.vulnerabilities.some(v => v.severity === 'high')) {
    return 'high';
  }

  // Medium: Minor version update
  if (pkg.updateType === 'minor') {
    return 'medium';
  }

  // Low: Patch update
  return 'low';
}
```

### Step 5: Generate Update Recommendations

Create prioritized update plan:

```
📦 Dependency Audit Report
Generated: <timestamp>

═══════════════════════════════════════════════════

🚨 CRITICAL PRIORITY (Immediate Action Required)

1. axios: 0.21.1 → 1.6.2 (MAJOR)
   Risk: CRITICAL
   Reason: 2 critical vulnerabilities (CVE-2023-XXXX, CVE-2023-YYYY)
   Impact: Remote code execution possible
   Action: npm install axios@latest
   Breaking Changes: Yes - Response interceptor API changed
   Migration: https://github.com/axios/axios/releases/v1.0.0

═══════════════════════════════════════════════════

⚠️  HIGH PRIORITY (Address This Week)

2. typescript: 4.9.5 → 5.3.3 (MAJOR)
   Risk: HIGH
   Reason: Major version update with breaking changes
   Impact: Type checking behavior changes, some syntax updates required
   Action: npm install -D typescript@latest
   Breaking Changes: Yes - Stricter checks, removed legacy features
   Migration: https://devblogs.microsoft.com/typescript/announcing-typescript-5-0/
   Note: Review all .ts files for type errors after update

3. react: 17.0.2 → 18.2.0 (MAJOR)
   Risk: HIGH
   Reason: Major version update
   Impact: Concurrent rendering, automatic batching, new APIs
   Action: npm install react@latest react-dom@latest
   Breaking Changes: Limited - Mostly additive
   Migration: https://react.dev/blog/2022/03/29/react-v18
   Note: Update to React 18 rendering API required

═══════════════════════════════════════════════════

📋 MEDIUM PRIORITY (Plan for Next Sprint)

4. eslint: 8.45.0 → 8.56.0 (MINOR)
   Risk: MEDIUM
   Reason: Minor version update
   Impact: New rules available, bug fixes
   Action: npm install -D eslint@latest
   Breaking Changes: None
   Note: Review .eslintrc for new rules

5. vitest: 0.34.0 → 1.1.0 (MAJOR)
   Risk: MEDIUM
   Reason: Major version but well-documented migration
   Impact: API improvements, better performance
   Action: npm install -D vitest@latest
   Breaking Changes: Yes - Config changes, some API renames
   Migration: https://vitest.dev/guide/migration.html

═══════════════════════════════════════════════════

✅ LOW PRIORITY (Can Wait)

6. prettier: 3.0.3 → 3.1.1 (MINOR)
   Risk: LOW
   Reason: Patch/minor update
   Impact: Formatting improvements
   Action: npm install -D prettier@latest

7. lodash: 4.17.20 → 4.17.21 (PATCH)
   Risk: LOW
   Reason: Patch update
   Impact: Bug fixes
   Action: npm install lodash@latest

═══════════════════════════════════════════════════

📊 Summary

Total Outdated: 7 packages
Critical: 1 (security vulnerabilities)
High: 2 (major version updates)
Medium: 2 (minor updates or major with good migration)
Low: 2 (patch updates)

Security Vulnerabilities:
- Critical: 2
- High: 0
- Moderate: 1
- Low: 0

Recommended Action Plan:
1. TODAY: Update axios (critical vulnerabilities)
2. THIS WEEK: Update TypeScript and React (plan for breaking changes)
3. NEXT SPRINT: Update ESLint and Vitest
4. ANYTIME: Update Prettier and lodash

═══════════════════════════════════════════════════

🎯 Quick Commands

# Critical updates only
npm install axios@latest

# All security fixes
npm audit fix

# Safe updates (minor + patch)
npx npm-check-updates -u --target minor
npm install

# Check what would change (dry run)
npx npm-check-updates
```

### Step 6: Warn About Major Version Updates

For major version updates, provide specific warnings:

```
⚠️  WARNING: Major Version Update Detected

Package: typescript 4.9.5 → 5.3.3

Breaking Changes Checklist:
- [ ] Review TypeScript 5.0 release notes
- [ ] Update tsconfig.json if needed
- [ ] Run typecheck: npm run typecheck
- [ ] Fix any new type errors
- [ ] Test build process
- [ ] Run full test suite
- [ ] Review for deprecated features

Recommended Approach:
1. Create a feature branch: git checkout -b upgrade/typescript-5
2. Update package: npm install -D typescript@latest
3. Fix type errors incrementally
4. Run tests after each fix
5. Merge when all tests pass

Rollback Plan:
If issues occur: npm install -D typescript@4.9.5
```

## Update Strategies

### Strategy 1: Conservative (Recommended)

```bash
# Only patch and minor updates
npx npm-check-updates -u --target minor
npm install
npm test
```

### Strategy 2: Aggressive (With Testing)

```bash
# Update everything
npx npm-check-updates -u
npm install

# Run full test suite
npm run typecheck
npm test
npm run build

# If anything fails, rollback and update incrementally
```

### Strategy 3: Security-Focused

```bash
# Fix vulnerabilities only
npm audit fix

# If audit fix requires breaking changes
npm audit fix --force  # ⚠️  May introduce breaking changes

# Verify after
npm test
```

### Strategy 4: Incremental (Safest)

```bash
# Update one package at a time
npm install <package>@latest
npm test

# If tests pass, commit
git add package.json package-lock.json
git commit -m "⬆️ chore: Update <package> to <version>"

# Move to next package
```

## Peer Dependency Checking

```bash
# Check for peer dependency warnings
npm install 2>&1 | grep -i "peer"

# Example output:
# npm WARN eslint-config-airbnb@19.0.4 requires a peer of eslint@^7.32.0 but none is installed
```

Fix peer dependency issues:

```bash
# Install missing peer dependency
npm install -D eslint@^7.32.0

# Or upgrade to satisfy peer dependency
npm install -D eslint@latest
```

## Breaking Change Detection

Automatically check for breaking changes:

```bash
# Get changelog URL
REPO=$(npm view <package> repository.url | sed 's/git+//' | sed 's/.git$//')

# Common changelog locations:
# - ${REPO}/blob/main/CHANGELOG.md
# - ${REPO}/releases
# - ${REPO}/blob/main/HISTORY.md

# Search for "BREAKING" in recent releases
curl -s "${REPO}/releases" | grep -i "breaking" -A 5
```

## Commands Reference

```bash
# Check outdated packages
npm outdated
npm outdated --json

# Security audit
npm audit
npm audit --json
npm audit fix
npm audit fix --force

# Update packages
npm install <package>@latest
npm install <package>@<version>

# Interactive updates
npx npm-check-updates
npx npm-check-updates -u
npx npm-check-updates -i

# Check package info
npm view <package> versions
npm view <package> version
npm view <package> homepage
npm view <package> repository

# List installed packages
npm list --depth=0
npm list <package>
```

## Integration with `/startWork`

When `/startWork` command runs:

```bash
# Automatically check for outdated packages
npm outdated --json > /tmp/outdated.json

# Parse and warn about major versions
MAJOR_UPDATES=$(jq -r 'to_entries[] | select(.value.type == "major") | .key' /tmp/outdated.json)

if [ -n "$MAJOR_UPDATES" ]; then
  echo "⚠️  Warning: Major version updates available:"
  echo "$MAJOR_UPDATES"
  echo ""
  echo "Consider running dependency audit before starting new work."
  echo "Run: dependency-auditor"
fi
```

## Best Practices

1. **Before starting work**:
   - Run dependency audit
   - Address critical/high security issues
   - Plan major updates separately

2. **During development**:
   - Avoid updating dependencies mid-feature
   - Create separate branches for major updates

3. **Regular maintenance**:
   - Weekly: Check for security updates
   - Monthly: Review outdated packages
   - Quarterly: Plan major version updates

4. **After updates**:
   - Run full test suite
   - Check build process
   - Verify in development environment
   - Test critical user flows

## Troubleshooting

### Issue: npm audit fix breaks build

```bash
# Rollback
git checkout package.json package-lock.json
npm install

# Fix incrementally
npm audit --json | jq -r '.vulnerabilities | to_entries[] | "\(.key): \(.value.severity)"'

# Update specific packages only
npm install <package>@<safe-version>
```

### Issue: Peer dependency conflicts

```bash
# Use --legacy-peer-deps (temporary fix)
npm install --legacy-peer-deps

# Better: Update packages to satisfy peer deps
npm install <package>@latest
```

### Issue: Package not found after update

```bash
# Clear npm cache
npm cache clean --force

# Remove and reinstall
rm -rf node_modules package-lock.json
npm install
```

## Example Output

```
User: "Check my dependencies"

Skill:
1. Runs: npm outdated --json
2. Runs: npm audit --json
3. Analyzes results
4. Generates report (shown above)
5. Provides recommendations

"📦 Found 7 outdated packages

⚠️  CRITICAL: axios has 2 critical vulnerabilities!
   Update immediately: npm install axios@latest

Detailed report generated above.
Would you like me to update the critical package now? (y/n)"
```
