---
name: Checkpoint Validator
description: Smart test selection and incremental typecheck for development checkpoints. Use when validating changes, running tests at checkpoints, verifying code quality during development, or when user mentions "checkpoint", "verify checkpoint", "validate changes", "run tests for changed files".
allowed-tools: Bash, Read, Grep, Glob
---

# Checkpoint Validator

This skill performs intelligent validation at development checkpoints by running only affected tests and incremental type checking.

## Purpose

Save ~70% of test time by running only tests related to changed files instead of the full test suite.

## Core Capabilities

1. **Smart Test Selection**: Identify and run only tests affected by changes
2. **Incremental Type Check**: Fast TypeScript validation
3. **Git Diff Analysis**: Understand what files changed
4. **Fast Validation Reports**: Clear summary of validation results

## Instructions

### Step 1: Identify Changed Files

```bash
# Get list of changed files since last commit
git diff --name-only HEAD

# Or for staged changes
git diff --staged --name-only

# Or compare to specific commit
git diff --name-only <commit-hash> HEAD
```

### Step 2: Analyze Affected Test Files

For each changed file, find related test files:

```bash
# Pattern 1: Co-located tests (file.ts → file.test.ts)
# Pattern 2: Mirror structure (src/foo.ts → tests/foo.test.ts)
# Pattern 3: Named pattern (foo.ts → foo.spec.ts)
```

**Logic**:
- If `src/auth/middleware.ts` changed → run `src/auth/middleware.test.ts` or `tests/auth/middleware.test.ts`
- If `api/routes/users.ts` changed → run `api/routes/users.test.ts` or `tests/routes/users.test.ts`
- If test file itself changed → run that test
- If shared utility changed → identify importers and run their tests

### Step 3: Run Incremental TypeCheck

```bash
# Fast incremental check (preferred)
npx tsc --noEmit --incremental

# Or use project's typecheck script
npm run typecheck
```

### Step 4: Run Targeted Tests

```bash
# Example with vitest (run specific test files)
npx vitest run <test-file-1> <test-file-2> <test-file-3>

# Example with jest
npx jest <test-file-1> <test-file-2>

# If no specific tests found, run all tests
npm test
```

### Step 5: Generate Validation Report

Create a concise summary:

```
✓ CHECKPOINT VALIDATION PASSED

Files Changed: 3
- src/auth/middleware.ts
- src/auth/types.ts
- src/utils/jwt.ts

Type Check: ✓ PASS (1.2s)

Tests Run: 2 test files (8 tests total)
- src/auth/middleware.test.ts ✓ PASS (5/5 tests)
- src/utils/jwt.test.ts ✓ PASS (3/3 tests)

Total Time: 3.4s (vs ~24s for full suite)
Time Saved: 85%

Ready to commit!
```

## Test Selection Algorithm

```typescript
function findAffectedTests(changedFiles: string[]): string[] {
  const testFiles = new Set<string>();

  for (const file of changedFiles) {
    // Skip if already a test file
    if (isTestFile(file)) {
      testFiles.add(file);
      continue;
    }

    // Pattern 1: Co-located test
    const colocated = file.replace(/\.ts$/, '.test.ts');
    if (exists(colocated)) {
      testFiles.add(colocated);
    }

    // Pattern 2: Mirror in tests/ directory
    const mirrored = file.replace(/^src\//, 'tests/').replace(/\.ts$/, '.test.ts');
    if (exists(mirrored)) {
      testFiles.add(mirrored);
    }

    // Pattern 3: .spec.ts variant
    const spec = file.replace(/\.ts$/, '.spec.ts');
    if (exists(spec)) {
      testFiles.add(spec);
    }

    // Pattern 4: Check for importers (if shared utility)
    if (isSharedUtility(file)) {
      const importers = findImporters(file);
      for (const importer of importers) {
        testFiles.add(...findAffectedTests([importer]));
      }
    }
  }

  return Array.from(testFiles);
}
```

## Edge Cases

### Case 1: No Tests Found
If no specific tests found for changed files:
- Warn the user
- Recommend adding tests
- Run full test suite as fallback

### Case 2: Shared Utilities Changed
If changed files are imported by many files:
- Identify all importers using `grep` or import analysis
- Run tests for all affected files
- May need to run more extensive test suite

### Case 3: Type Definitions Changed
If `.d.ts` or type files changed:
- Run full typecheck (types affect everything)
- Consider running broader test suite

### Case 4: Config Files Changed
If `tsconfig.json`, `vitest.config.ts`, etc. changed:
- Run full test suite (config affects all tests)
- Run full typecheck

## Performance Targets

| Validation Type | Target Time | vs Full Suite |
|----------------|-------------|---------------|
| Incremental TypeCheck | < 2s | ~80% faster |
| Targeted Tests (1-3 files) | < 5s | ~85% faster |
| Targeted Tests (4-10 files) | < 15s | ~60% faster |
| Full Suite (fallback) | Variable | Baseline |

## Integration with Workflows

### During `/startWork`
After each micro checkpoint:
1. Make code changes
2. **Run Checkpoint Validator**
3. Fix any issues
4. Commit with conventional commit message
5. Proceed to next checkpoint

### Before Committing
```bash
# User makes changes
# Stages changes: git add .

# Run validation
checkpoint-validator

# If PASS → commit
# If FAIL → fix issues, re-validate
```

## Failure Handling

If validation fails:

```
✗ CHECKPOINT VALIDATION FAILED

Type Check: ✗ FAIL (3 errors)
  src/auth/middleware.ts:45:12 - Type 'string' not assignable to 'number'
  src/auth/middleware.ts:67:5 - Property 'userId' missing in type

Tests Run: 2 test files (8 tests total)
- src/auth/middleware.test.ts ✗ FAIL (3/5 tests)
  ✗ should validate token expiry
  ✗ should reject invalid signature
- src/utils/jwt.test.ts ✓ PASS (3/3 tests)

Action Required:
1. Fix type errors in src/auth/middleware.ts
2. Fix failing tests in src/auth/middleware.test.ts
3. Re-run validation

DO NOT COMMIT until validation passes!
```

## Commands Reference

```bash
# Check what changed
git diff --name-only HEAD

# Type check
npx tsc --noEmit --incremental

# Run specific test file
npx vitest run path/to/test.test.ts

# Run multiple test files
npx vitest run test1.test.ts test2.test.ts

# Run all tests (fallback)
npm test

# Check test coverage for specific files
npx vitest run --coverage path/to/test.test.ts
```

## Example Workflow

```
User: "Validate my checkpoint"

Skill:
1. Runs: git diff --name-only HEAD
   → Found: src/auth/middleware.ts, src/auth/types.ts

2. Identifies tests:
   → src/auth/middleware.test.ts (exists)
   → src/auth/types.test.ts (not found, skipped)

3. Runs: npx tsc --noEmit --incremental
   → ✓ No type errors

4. Runs: npx vitest run src/auth/middleware.test.ts
   → ✓ All 5 tests passed

5. Reports:
   "✓ Checkpoint validation passed!
    Time: 3.2s (saved ~21s vs full suite)
    Ready to commit."
```
