---
name: GitHub Project Work Planner
description: Intelligent GitHub Projects v2 integration for work planning, prioritization, and sprint management. Use when planning work, checking project status, syncing with GitHub Projects, deciding what to work on next, creating sprint plans, or when user mentions "plan work", "GitHub project", "what should I work on", "sprint planning", "project board", "sync project".
allowed-tools: Bash, Read, Write, Grep, Glob
---

# GitHub Project Work Planner

Transform your GitHub Projects board into an intelligent work orchestrator with automatic prioritization, dependency detection, and seamless integration with your development workflow.

## Core Capabilities

1. **Intelligent Work Planning**: Analyze issues and suggest optimal work sequence
2. **Dependency Detection**: Identify blockers and unblock opportunities
3. **Auto-Generate TaskFindings**: Create planning docs from GitHub issues
4. **Two-Way Sync**: Update GitHub Projects as work progresses
5. **Sprint Planning**: Capacity-aware weekly planning
6. **Progress Visualization**: Terminal progress bars and reports

## Prerequisites

- GitHub CLI (`gh`) installed and authenticated
- GitHub Projects v2 board set up
- Issues labeled with complexity/effort/priority (recommended)

## Instructions

### Step 1: Fetch Project Data

```bash
# List all projects for the repo
gh project list --owner <org-or-user>

# View project items (issues/PRs)
gh project item-list <project-number> --owner <org-or-user> --format json

# Get specific issue details
gh issue view <issue-number> --json number,title,body,labels,state,assignees
```

### Step 2: Analyze Dependencies

Parse issue bodies for dependency markers:

```markdown
## Dependencies
Blocked by: #45
Blocks: #47, #48, #52
Related to: #50
```

Build a dependency graph to identify:
- Critical blockers (issues blocking multiple downstream tasks)
- Quick wins (no dependencies, high impact)
- Work sequences (optimal order to tackle issues)

### Step 3: Calculate Priority Scores

```typescript
interface Issue {
  number: number;
  complexity: 1-5;  // From labels: complexity-1 to complexity-5
  priority: 'low' | 'medium' | 'high';  // From labels
  blocks: number[];  // Issue numbers this blocks
  blockedBy: number[];  // Issue numbers blocking this
  estimate: number;  // Hours, from labels: estimated-Xh
}

function calculatePriorityScore(issue: Issue): number {
  let score = 0;

  // Priority weight
  if (issue.priority === 'high') score += 3;
  if (issue.priority === 'medium') score += 2;
  if (issue.priority === 'low') score += 1;

  // Blocker weight (unblocking is valuable)
  score += issue.blocks.length * 2;

  // Complexity weight (favor quick wins slightly)
  if (issue.complexity === 1) score += 1;

  // No blockers (can start immediately)
  if (issue.blockedBy.length === 0) score += 1;

  return score;
}
```

### Step 4: Generate Work Plan

Create a capacity-aware plan (assume 40h/week unless specified):

```
🎯 Recommended Work Sequence (40h capacity)

1. #51 Add dark mode toggle ⚡ QUICK WIN
   ⏱️  3h | 🎚️  1/5 | 🏷️  feature, frontend
   Why first: Quick win, high user impact, no blockers

2. #45 Refactor auth middleware 🚧 UNBLOCKS 3 ISSUES
   ⏱️  8h | 🎚️  3/5 | 🏷️  refactor, infrastructure
   Why next: Unblocks #47, #48, #52 for future sprints

3. #53 Fix payment webhook timeout 🔥 HIGH PRIORITY
   ⏱️  6h | 🎚️  2/5 | 🏷️  bug, high-priority
   Why: Production issue affecting revenue

Total: 17h / 40h (43% capacity utilization)
```

### Step 5: Create TaskFindings from Issue

When user selects an issue to work on:

```bash
# Fetch full issue details
gh issue view <issue-number> --json number,title,body,labels,milestone

# Parse acceptance criteria from issue body
# Create taskFindings.md in appropriate directory
```

TaskFindings template:

```markdown
# Purpose
<Extracted from issue title/body>

## Original Ask
<Issue body content>

## Complexity and the reason behind it
Complexity: <from complexity-X label>/5
Reason: <Explain why this complexity score>

## Architectural changes required
<Parse from issue or set to "None">

## Backend changes required
<List from issue or infer from labels>

## Frontend changes required
<List from issue or infer from labels>

## Acceptance Criteria
<Extract from issue body checkboxes>
- [ ] Criterion 1
- [ ] Criterion 2

## Validation
<Generate test plan from acceptance criteria>

---
*Auto-generated from GitHub Issue #<number>*
*Last synced: <timestamp>*
```

### Step 6: Two-Way Sync

**When work starts** (`/startWork`):
```bash
# Move issue to "In Progress"
gh issue edit <issue-number> --add-label "in-progress"

# Assign to self
gh issue edit <issue-number> --add-assignee @me

# Add comment
gh issue comment <issue-number> --body "🚀 Started work on this issue"
```

**When work completes** (`/completeWork`):
```bash
# Move to "Done" (or close if appropriate)
gh issue edit <issue-number> --remove-label "in-progress" --add-label "ready-for-review"

# Add completion comment
gh issue comment <issue-number> --body "✅ Work completed and validated

Validation Summary:
- All acceptance criteria met
- Tests: PASS
- Ready for code review

Files changed: <count>
Lines: +<additions> / -<deletions>

🤖 Auto-updated by WorkPlanner Skill"

# Check for unblocked issues
# If this issue blocked others, notify
```

### Step 7: Sprint Progress Reports

```bash
# Get all issues in current sprint
gh issue list --label "sprint-current" --json number,title,state,labels

# Calculate progress metrics
```

Generate report:

```
📊 Sprint Progress Report
Week of <date-range>

═══════════════════════════════════════════════════

📈 Velocity Tracking
Planned:    40h | ████████████████████████████████ 100%
Completed:  28h | ████████████████████░░░░░░░░░░░░  70%
Remaining:  12h | ████████░░░░░░░░░░░░░░░░░░░░░░░░  30%

Status: ON TRACK ✅

═══════════════════════════════════════════════════

✅ Completed (4 issues, 17h)
#51 Add dark mode toggle (3h) - Frontend
#53 Fix payment webhook (6h) - Backend Bug

🏃 In Progress (2 issues, 14h estimated)
#45 Refactor auth middleware (8h, 75% complete)
#47 Add OAuth social login (6h, 25% complete)

⏸️  Blocked (1 issue)
#48 Add MFA support - Blocked by #45 (in progress)

📝 Ready (5 issues, 33h)
```

## Label Conventions

For best results, use these label patterns:

### Complexity
- `complexity-1`: Trivial (< 2h)
- `complexity-2`: Simple (2-4h)
- `complexity-3`: Medium (4-8h)
- `complexity-4`: Complex (1-2 days)
- `complexity-5`: Very complex (> 2 days)

### Estimates
- `estimated-1h`, `estimated-3h`, `estimated-8h`, `estimated-1d`, etc.

### Priority
- `priority-low`
- `priority-medium`
- `priority-high`

### Impact
- `impact-low`
- `impact-medium`
- `impact-high`

### Type
- `bug`, `feature`, `refactor`, `docs`, `test`, `chore`

## Dependency Parsing

Support multiple formats in issue bodies:

```markdown
<!-- Format 1: Simple -->
Blocked by: #45
Blocks: #47, #48

<!-- Format 2: Detailed -->
## Dependencies
- Blocked by: #45 (auth refactor)
- Blocks: #47 (OAuth), #48 (MFA)
- Related: #50 (API keys)

<!-- Format 3: Checkbox -->
## Prerequisites
- [ ] #45 Auth refactor must be complete
- [ ] Backend deployment to staging
```

Parse using regex or structured parsing.

## Conflict Detection

When planning work, check for file conflicts:

```bash
# Get files touched by in-progress issues
# Warn if multiple issues touch the same files

⚠️  Warning: Multiple issues touch auth.ts
- #45 In Progress: Refactoring auth middleware
- #48 Ready: Adding MFA support

Recommendation: Complete #45 before starting #48 to avoid merge conflicts
```

## Velocity Learning

Track historical data to improve estimates:

```typescript
// Store completion data
interface CompletionData {
  issueNumber: number;
  estimatedHours: number;
  actualHours: number;
  complexity: number;
  type: string;
}

// Calculate average variance by complexity
function analyzeVelocity(history: CompletionData[]) {
  const byComplexity = groupBy(history, 'complexity');

  for (const [complexity, items] of byComplexity) {
    const avgEstimated = average(items.map(i => i.estimatedHours));
    const avgActual = average(items.map(i => i.actualHours));
    const variance = (avgActual - avgEstimated) / avgEstimated;

    console.log(`Complexity ${complexity}: avg ${variance > 0 ? 'over' : 'under'}run of ${Math.abs(variance * 100).toFixed(0)}%`);
  }
}
```

## Integration Points

### With `/codePlanner`
```bash
# User runs: /codePlanner 123
# Skill automatically:
1. Fetches issue #123 from GitHub
2. Creates taskFindings.md
3. Updates issue status to "planning"
```

### With `/startWork`
```bash
# User runs: /startWork feature-name
# Skill automatically:
1. Finds corresponding GitHub issue
2. Updates status to "in-progress"
3. Assigns to user
4. Adds start comment
```

### With `/completeWork`
```bash
# User runs: /completeWork feature-name
# Skill automatically:
1. Reads validation results
2. Updates issue with completion comment
3. Moves to "ready-for-review" or "done"
4. Checks for unblocked downstream issues
```

## Example Workflow

```
User: "Plan my work for this week"

Skill:
1. Fetches GitHub Project "Q1 2025 Sprint"
2. Analyzes 23 issues
3. Builds dependency graph
4. Calculates priority scores
5. Generates capacity-aware plan:

"🎯 Recommended Work Sequence (40h capacity)

1. #51 Dark mode toggle ⚡ (3h)
2. #45 Auth refactor 🚧 (8h) - Unblocks 3 issues
3. #53 Payment fix 🔥 (6h) - High priority
...

Which issue would you like to start? (#): "

User: "51"

Skill:
1. Fetches issue #51 details
2. Creates taskNotes/dark-mode-toggle/taskFindings.md
3. Updates GitHub: moves to "In Progress", assigns to user
4. Reports: "✅ Ready to start! Run: /startWork dark-mode-toggle"
```

## Error Handling

```bash
# If gh CLI not installed
command -v gh >/dev/null 2>&1 || {
  echo "❌ GitHub CLI not found. Install with: brew install gh"
  exit 1
}

# If not authenticated
gh auth status || {
  echo "❌ Not authenticated. Run: gh auth login"
  exit 1
}

# If project not found
gh project list --owner <owner> | grep -q "<project>" || {
  echo "❌ Project not found. Check owner and project name."
  exit 1
}
```

## Commands Reference

```bash
# List projects
gh project list --owner <org>

# View project items
gh project item-list <project-id> --format json

# View issue
gh issue view <number> --json number,title,body,labels,state

# Edit issue
gh issue edit <number> --add-label "label" --add-assignee @me

# Comment on issue
gh issue comment <number> --body "message"

# List issues with filters
gh issue list --label "sprint-current" --state open --json number,title
```
