---
name: Project Scaffolder
description: Autonomous project scaffolding with framework-specific templates and best practices. Auto-applies when user mentions creating projects, frameworks (Hono, Next.js, CDK), or says "create", "scaffold", "initialize", "setup", "new project". Handles complete project setup from git init to deployment-ready structure.
allowed-tools: Bash, Write, Read, Grep, Glob, MultiEdit, mcp__jina__read_webpage, mcp__jina__search_web, mcp__cloudflare-docs__search_cloudflare_documentation
---

# Project Scaffolder Skill

Autonomous project scaffolding that detects intent and creates production-ready project structures with framework-specific best practices, tooling, and configuration.

## When This Skill Auto-Applies

This skill automatically scaffolds projects when user mentions:

**Direct Project Creation:**
- "Create a new Next.js app called [name]"
- "Scaffold a Hono API project"
- "Initialize a CDK app"
- "Set up a new [framework] project"
- "Start a new [framework] application"

**Framework Names:**
- Hono, Next.js, CDK, React, Vite
- Cloudflare Workers, AWS Lambda
- T3 Stack, tRPC

**Action Keywords:**
- create, scaffold, initialize, setup, start, new
- Combined with: project, app, application, API, service

**Examples:**
- "Let's create a Hono API on Cloudflare Workers"
- "I need a new Next.js app"
- "Scaffold a CDK project for AWS"
- "Create a full-stack app with T3 Stack"

## Supported Frameworks

### Backend APIs
- **Hono** (Cloudflare Workers, Bun, Node.js, Deno)
- **CDK** (AWS Infrastructure as Code)

### Frontend Applications
- **Next.js** (Cloudflare Workers, Vercel, Static)
- **React** (SPA, Vite)

### Full-stack
- **T3 Stack** (Next.js + tRPC + Prisma)
- **Hono + Next.js** (Monorepo setup)

## Framework Selection Guide

| Use Case | Framework | Command |
|----------|-----------|---------|
| REST API on Cloudflare Workers | Hono | `npm create cloudflare@latest -- --framework=hono` |
| REST API on Bun/Node | Hono | `bun create hono` or `npm create hono@latest` |
| Frontend on Cloudflare | Next.js (Pages) | `npm create cloudflare@latest -- --framework=next` |
| AWS Infrastructure | CDK | `npx cdk init app --language=typescript` |
| Full-stack SPA | Vite + React | `npm create vite@latest` |

## User Preferences (from CLAUDE.md)

**Backend APIs:**
- **STRONGLY PREFER Hono Framework** for all API projects
  - Cloudflare Workers: `npm -y create cloudflare@latest [name] -- --framework=hono --git --deploy=false`
  - Bun/Deno/Node: Also use Hono

**Frontend:**
- Next.js for Cloudflare Workers: `npm -y create cloudflare@latest [name] -- --framework=next --platform=workers -y`

**Infrastructure:**
- CDK with TypeScript: `npx cdk init app --language=typescript`

**Database:**
- Prefer Postgres for AWS projects
- Open to DynamoDB if required
- D1 for Cloudflare

**Testing:**
- Use Vitest for unit tests

**Package Management:**
- Use package manager commands (npm, bun, etc.) to install dependencies

## Auto-Detection & Smart Defaults

When user provides minimal info, make intelligent assumptions:

**User says: "Create an API"**
→ Default to: Hono on Cloudflare Workers (user preference)

**User says: "Create a frontend app"**
→ Default to: Next.js on Cloudflare Pages (user preference)

**User says: "Create an AWS project"**
→ Default to: CDK with TypeScript (user preference)

**User says: "Create a full-stack app"**
→ Ask: Monorepo with Hono + Next.js or T3 Stack?

**If unclear:**
- Ask minimal clarifying questions
- Suggest user's preferred stack first
- Proceed with smart defaults

## Instructions

### Step 1: Understand Requirements

Ask clarifying questions if not specified:

1. **Framework**: Which framework? (Hono, Next.js, CDK, etc.)
2. **Platform**: Where will it run? (Cloudflare Workers, AWS Lambda, Vercel, etc.)
3. **Database**: Need database? (Postgres, D1, DynamoDB, none)
4. **Auth**: Need authentication? (Yes/No)
5. **Project name**: What to call the project?

### Step 2: Initialize Git Repository

```bash
# Create project directory
mkdir <project-name>
cd <project-name>

# Initialize git
git init

# Create .gitignore (framework-specific)
```

### Step 3: Scaffold Framework

#### For Hono API (Cloudflare Workers)

```bash
npm create cloudflare@latest <project-name> -- --framework=hono --git --deploy=false

# Results in:
# - Hono app with routes
# - wrangler.jsonc configuration
# - TypeScript configured
# - Git initialized
# - Dependencies installed
```

#### For Next.js (Cloudflare Pages)

```bash
npm create cloudflare@latest <project-name> -- --framework=next --platform=workers

# Results in:
# - Next.js app with App Router
# - Cloudflare Pages configuration
# - TypeScript configured
# - Tailwind CSS (optional)
```

#### For CDK (AWS)

```bash
mkdir <project-name>
cd <project-name>
npx cdk init app --language=typescript

# Results in:
# - CDK app structure
# - Sample stack
# - TypeScript configured
# - Jest for testing
```

#### For Hono API (Bun)

```bash
bun create hono <project-name>

# Select runtime: bun
# Results in:
# - Hono app optimized for Bun
# - Fast development server
```

### Step 4: Add Environment Configuration

Create `.env.example`:

```bash
# .env.example template
cat > .env.example <<'EOF'
# Application
NODE_ENV=development
PORT=3000

# Database (if applicable)
DATABASE_URL=postgresql://user:pass@localhost:5432/dbname

# Auth (if applicable)
JWT_SECRET=your-secret-key-here
SESSION_SECRET=your-session-secret

# API Keys (if applicable)
API_KEY=your-api-key

# Cloudflare (if applicable)
CLOUDFLARE_ACCOUNT_ID=
CLOUDFLARE_API_TOKEN=
EOF
```

Create `.env`:

```bash
cp .env.example .env
echo "✅ Created .env file - update with your actual values"
```

### Step 5: Configure Package Scripts

Add useful scripts to `package.json`:

```json
{
  "scripts": {
    "dev": "framework-specific-dev-command",
    "build": "framework-specific-build",
    "test": "vitest",
    "test:watch": "vitest watch",
    "typecheck": "tsc --noEmit",
    "lint": "eslint .",
    "format": "prettier --write .",
    "prepare": "husky install"
  }
}
```

### Step 6: Add Testing Setup

Install testing dependencies:

```bash
# For Hono/Next.js/React projects
npm install -D vitest @vitest/ui

# Create vitest.config.ts
cat > vitest.config.ts <<'EOF'
import { defineConfig } from 'vitest/config';

export default defineConfig({
  test: {
    globals: true,
    environment: 'node', // or 'jsdom' for frontend
    coverage: {
      provider: 'v8',
      reporter: ['text', 'json', 'html'],
    },
  },
});
EOF
```

### Step 7: Add Linting & Formatting

```bash
# Install ESLint & Prettier
npm install -D eslint prettier eslint-config-prettier

# Create .eslintrc.json (basic config)
cat > .eslintrc.json <<'EOF'
{
  "extends": ["eslint:recommended", "prettier"],
  "env": {
    "node": true,
    "es2022": true
  },
  "parserOptions": {
    "ecmaVersion": 2022,
    "sourceType": "module"
  }
}
EOF

# Create .prettierrc
cat > .prettierrc <<'EOF'
{
  "semi": true,
  "trailingComma": "es5",
  "singleQuote": true,
  "printWidth": 80,
  "tabWidth": 2
}
EOF
```

### Step 8: Create README.md

Generate framework-specific README:

```markdown
# <Project Name>

<Brief description>

## Tech Stack

- **Framework**: <Hono/Next.js/CDK>
- **Runtime**: <Cloudflare Workers/Bun/Node.js>
- **Database**: <PostgreSQL/D1/None>
- **Testing**: Vitest

## Getting Started

### Prerequisites

- Node.js 20+ (or Bun 1.0+)
- <Other prerequisites>

### Installation

\`\`\`bash
npm install
cp .env.example .env
# Update .env with your values
\`\`\`

### Development

\`\`\`bash
npm run dev
\`\`\`

### Testing

\`\`\`bash
npm test
\`\`\`

### Build

\`\`\`bash
npm run build
\`\`\`

### Deployment

<Framework-specific deployment instructions>

## Project Structure

\`\`\`
<project-name>/
├── src/
│   ├── index.ts        # Entry point
│   ├── routes/         # API routes
│   └── utils/          # Utilities
├── tests/              # Test files
├── .env.example        # Environment template
├── package.json
└── tsconfig.json
\`\`\`

## Available Scripts

- \`npm run dev\` - Start development server
- \`npm test\` - Run tests
- \`npm run build\` - Build for production
- \`npm run typecheck\` - Check TypeScript types
- \`npm run lint\` - Lint code

## License

MIT
```

### Step 9: Initial Commit

```bash
git add .
git commit -m "🎉 chore: Initial project scaffold with <framework>"
```

### Step 10: Guide Next Steps

Print helpful next steps:

```
✅ Project scaffolded successfully!

📁 Project: <project-name>
🛠️  Framework: <framework>
📦 Dependencies: Installed

Next Steps:
1. cd <project-name>
2. Update .env with your configuration
3. Run: npm run dev
4. Start building!

Optional:
- Create GitHub repository: gh repo create <project-name> --public
- Push code: git remote add origin <url> && git push -u origin main
- Deploy to Cloudflare: npm run deploy (if applicable)

Happy coding! 🚀
```

## Framework-Specific Templates

### Hono API Template Structure

```
hono-api/
├── src/
│   ├── index.ts           # App entry
│   ├── routes/
│   │   ├── index.ts       # Route aggregator
│   │   ├── health.ts      # Health check
│   │   └── api/
│   │       └── v1/        # Versioned API
│   ├── middleware/
│   │   ├── auth.ts        # Auth middleware
│   │   ├── cors.ts        # CORS config
│   │   └── logger.ts      # Request logging
│   ├── utils/
│   │   └── jwt.ts         # JWT utilities
│   └── types/
│       └── index.ts       # Type definitions
├── tests/
│   └── routes/
│       └── health.test.ts
├── wrangler.jsonc         # Cloudflare config
├── .env.example
└── package.json
```

### Next.js Template Structure

```
nextjs-app/
├── app/
│   ├── layout.tsx         # Root layout
│   ├── page.tsx           # Home page
│   └── api/               # API routes
│       └── hello/
│           └── route.ts
├── components/
│   └── ui/                # Reusable components
├── lib/
│   └── utils.ts           # Utilities
├── public/
├── .env.example
└── package.json
```

### CDK Template Structure

```
cdk-app/
├── bin/
│   └── app.ts             # CDK app entry
├── lib/
│   └── app-stack.ts       # Main stack
├── test/
│   └── app.test.ts        # Stack tests
├── cdk.json
└── package.json
```

## Best Practices

1. **Always include**:
   - `.gitignore` (comprehensive)
   - `.env.example` (never commit .env)
   - `README.md` (clear instructions)
   - TypeScript configuration
   - Testing setup

2. **Framework conventions**:
   - Hono: Use middleware pattern, route grouping
   - Next.js: App Router, Server Components
   - CDK: Separate stacks by concern

3. **Security**:
   - Never scaffold with hardcoded secrets
   - Always use `.env` for configuration
   - Include security headers in API responses

4. **Developer experience**:
   - Fast dev server startup
   - Hot reload configured
   - Clear error messages
   - Type safety everywhere

## Anti-Patterns to Avoid

❌ Scaffolding without understanding requirements
✅ Ask clarifying questions first

❌ Committing `.env` file
✅ Only commit `.env.example`

❌ Minimal README
✅ Comprehensive README with setup instructions

❌ No tests configured
✅ Include testing framework from start

❌ Manual dependency installation
✅ Use framework-specific create commands

## Troubleshooting

### Issue: npm create fails

```bash
# Clear npm cache
npm cache clean --force

# Update npm
npm install -g npm@latest

# Try again
```

### Issue: TypeScript errors after scaffold

```bash
# Reinstall dependencies
rm -rf node_modules package-lock.json
npm install

# Check tsconfig.json is valid
npx tsc --noEmit
```

### Issue: Port already in use

```bash
# Change port in .env or dev script
# Or kill process on port
lsof -ti:3000 | xargs kill -9
```


## Notes on This Skill

**Consolidated from:**
- Previous `/skills/scaffolder/SKILL.md` (framework templates and best practices)
- `/commands/scaffolder.md` (manual scaffolding command)
- `/agents/project-scaffolder.md` (autonomous scaffolding agent)

**Key improvements:**
- Fully autonomous - detects intent automatically
- Incorporates user preferences from CLAUDE.md
- Smart defaults reduce friction
- Complete end-to-end scaffolding
- No manual command invocation needed

**Original files archived:**
- `/commands/scaffolder.md` → `/commands/legacy/scaffolder.md`
- `/agents/project-scaffolder.md` → removed (consolidated here)

This skill addresses the user pain point: "scaffolding is clunky"

