---
name: Git Best Practices
description: Autonomous git operations with intelligent conflict resolution and safety checks. Use when git operations are needed - branch management, merging, syncing, committing, pushing. Auto-applies when user mentions git operations or during workflow phases.
allowed-tools: Bash(git:*), Bash(npm:*), Bash(yarn:*), Bash(test:*), Bash(build:*)
---

# Git Best Practices Skill

Autonomous git operations following best practices with intelligent conflict resolution, safety checks, and seamless integration with conventional-commit skill.

## When This Skill Auto-Applies

This skill automatically handles git operations when:
- User mentions git operations: "merge", "sync", "commit", "push", "branch"
- During /startWork: branch creation/switching
- During development: commits at checkpoints
- User says "commit and push"
- During /completeWork: final commits and push
- Cleanup operations: "clean branches", "delete merged branches"

## Core Operations

### 1. Branch Management

**Create/Switch Branch**
```bash
# Auto-detects branch operation needs
# Creates branch with naming convention: feature/issue-123-description
# Switches to branch safely (stashes if needed)
```

**Process:**
1. Check if branch exists locally
2. If not, check remote
3. Create from appropriate base (main/master)
4. Switch with safety checks
5. Update tracking

### 2. Smart Merge

**Auto-applies when:** User wants to merge branches

**Process:**
1. Fetch latest changes from origin
2. Create backup branch: `backup-merge-$(date +%s)`
3. Attempt merge with recursive strategy
4. If conflicts:
   - Analyze conflict patterns
   - Apply automated fixes for simple conflicts (imports, formatting)
   - Show remaining conflicts with context
   - Suggest resolution strategies
5. Run tests if available (npm test, yarn test)
6. Provide rollback: `git reset --hard backup-merge-<timestamp>`

**Safety:**
- Always create backup before merge
- Never auto-commit with conflicts
- Provide clear rollback instructions
- Test before finalizing

### 3. Branch Sync

**Auto-applies when:** Need to sync current branch with main/master

**Process:**
1. Stash uncommitted changes
2. Fetch origin
3. Identify base branch (main or master)
4. Rebase current branch onto base
5. Handle conflicts intelligently:
   - Show conflict files
   - Suggest conflict resolution
   - Provide `git rebase --abort` option
6. Restore stashed changes
7. Offer force-push-with-lease if requested

**Safety:**
- Create backup before rebase
- Stash preserves work-in-progress
- Warn about force-push implications
- Provide recovery commands

### 4. Commit Operations

**Auto-applies when:**
- User says "commit", "commit and push"
- Checkpoint reached during /startWork
- Before /completeWork finalization

**Process:**
1. Check for changes (git status)
2. Invoke conventional-commit skill for message generation
3. Stage appropriate files
4. Create commit with generated message
5. Show commit summary

**Integration:**
- Works seamlessly with conventional-commit skill
- Follows CLAUDE.md commit format rules
- Auto-commits at checkpoints
- Never commits secrets (.env, credentials.json, etc.)

### 5. Push Operations

**Auto-applies when:** User says "push", "commit and push"

**Process:**
1. Check branch tracking
2. Set upstream if needed (-u flag)
3. Check for uncommitted changes
4. Offer to commit first if changes exist
5. Push to remote
6. Handle push failures:
   - Suggest pull if behind
   - Suggest force-with-lease if diverged
   - Never force push to main/master

**Safety:**
- Never force push to protected branches
- Warn about force push implications
- Check for tracking branch
- Handle upstream configuration

### 6. Branch Cleanup

**Auto-applies when:** User wants to clean up branches

**Process:**
1. List branches merged into main/master
2. Identify stale branches (older than specified days)
3. Show current worktrees to avoid deletion
4. Confirm before deletion
5. Clean up remote tracking branches
6. Provide restore command for each deleted branch

**Safety:**
- Never delete current branch
- Never delete main/master
- Skip branches with worktrees
- Always provide restore commands
- Confirm before deletion

### 7. Hotfix Flow

**Auto-applies when:** Emergency production fix needed

**Process:**
1. Create hotfix branch from main/master
2. Cherry-pick specific commits if needed
3. Fast-track testing (critical tests only)
4. Merge to main and develop
5. Tag release
6. Push with urgency

**Safety:**
- Still requires tests to pass
- Tag for traceability
- Merge to both main and develop

### 8. Git Blame/History

**Auto-applies when:** User needs to investigate code history

**Process:**
1. Show file history with context
2. Identify commit that introduced line
3. Show commit details
4. Link to related issues/PRs

### 9. Gitignore Management

**Auto-applies when:** User wants to ignore files

**Process:**
1. Add patterns to .gitignore
2. Remove from tracking if already tracked
3. Test ignore patterns
4. Commit .gitignore changes

## Integration with Other Skills

### With conventional-commit skill:
- Generates commit messages automatically
- Follows emoji and type conventions
- Single-line format

### With code-review skill:
- Review happens BEFORE commit
- Blocks commit if critical issues
- Suggests fixes before committing

### With checkpoint-validator skill:
- Commits after successful checkpoints
- Test results inform commit decision

## Auto-Detection Patterns

Triggers when user says:
- "merge main into this branch"
- "sync with main"
- "commit and push"
- "create branch for issue-123"
- "clean up old branches"
- "push to remote"
- "rebase on main"
- "cherry-pick commit abc123"
- "blame this file"

## Safety Principles

1. **Always Create Backups**: Before destructive operations
2. **Never Force Without Confirmation**: Especially to main/master
3. **Provide Rollback Instructions**: For every risky operation
4. **Test Before Finalizing**: Run tests when available
5. **Stash Preserves Work**: Never lose uncommitted changes
6. **Warn About Implications**: Force push, rebase, etc.

## Error Handling

**Merge Conflicts:**
- Show files with conflicts
- Suggest resolution strategies
- Provide abort option
- Keep backup branch

**Push Failures:**
- Check if behind remote
- Suggest pull/rebase
- Offer force-with-lease if appropriate
- Never auto force-push

**Rebase Conflicts:**
- Show conflict details
- Suggest continue/skip/abort
- Preserve stashed changes
- Provide recovery path

## Workflow Integration

### During /startWork:
```
✓ Check if branch exists
✓ Create/switch to feature branch
✓ Ensure up-to-date with main
✓ Ready for development
```

### During Development:
```
✓ Auto-commit at checkpoints
✓ Generate commit messages via conventional-commit
✓ Handle branch switching if needed
```

### During /completeWork:
```
✓ Final commit with all changes
✓ Sync with main if needed
✓ Push to remote
✓ Ready for PR (or invoke pr-automation skill if available)
```

## Commands Consolidated

This skill consolidates these manual commands:
- /git:smart-merge
- /git:sync-branch
- /git:branch-cleanup
- /git:hotfix-flow
- /git:blame
- /git:find-commit
- /git:ignore

Original commands remain as fallback manual overrides if needed.

## Examples

**User: "commit and push"**
```
Skill auto-applies:
1. Checks for changes
2. Invokes conventional-commit for message
3. Stages files
4. Creates commit
5. Pushes to remote
```

**User: "merge main into this branch"**
```
Skill auto-applies:
1. Creates backup branch
2. Fetches latest main
3. Attempts merge
4. Handles conflicts if any
5. Runs tests
6. Provides status report
```

**User: "clean up old branches"**
```
Skill auto-applies:
1. Lists merged branches
2. Shows stale branches
3. Confirms deletion
4. Cleans up with safety
5. Provides restore commands
```

## Best Practices Applied

- Use `--force-with-lease` instead of `--force`
- Create backup branches before risky operations
- Rebase for cleaner history on feature branches
- Merge for integration to main branches
- Test before pushing
- Never commit secrets
- Semantic branch names
- Clear commit messages via conventional-commit

## Recovery Commands

Always provide these when applicable:
```bash
# Undo last commit (keep changes)
git reset --soft HEAD~1

# Restore from backup
git reset --hard backup-merge-<timestamp>

# Abort merge
git merge --abort

# Abort rebase
git rebase --abort

# Restore deleted branch
git checkout -b <branch-name> <commit-hash>
```

## Output Format

Provide clear, concise status:
```
✓ Branch created: feature/issue-123-dark-mode
✓ Committed: ✨ feat: Add dark mode toggle (abc123)
✓ Pushed to origin/feature/issue-123-dark-mode
✓ Ready for PR
```

For conflicts:
```
⚠️  Merge conflicts in 2 files:
  - src/components/Header.tsx
  - src/styles/theme.css

Suggested resolution:
1. Accept incoming changes for theme.css
2. Manual merge needed for Header.tsx

Abort: git merge --abort
```

## Notes

- This skill operates autonomously based on context
- Manual git commands still available as fallback
- Integrates with conventional-commit for messages
- Integrates with code-review before commits
- Follows safety-first principles
- Always provides escape hatches
